<?php defined('BASEPATH') OR exit('No direct script access allowed');

class PiketDaily extends CI_Controller
{
    /* ===== Konstanta ===== */
    private const WEEKDAY_JAM  = '17.00 - 23.00';
    private const SATURDAY_JAM = '12.00 - 17.00';
    private const JAM_REGEX    = '/^\d{2}\.\d{2}\s-\s\d{2}\.\d{2}$/';
    private const SIG_ORDER    = ['PIC','CHECK','CONFIRM','APPROVE'];

    public function __construct()
{
    parent::__construct();

    // Guard login + simpan URL yang diminta
    if (!$this->session->userdata('role')) {
        // butuh helper url -> sudah ada di bawah, tapi aman kalau dipanggil lagi
        $this->load->helper('url');

        // simpan URL yg sedang diakses (misal: PiketDaily/sign_pad/26/CONFIRM)
        $this->session->set_userdata('after_login_redirect', current_url());

        redirect('login');
    }

    if (function_exists('date_default_timezone_set')) {
        date_default_timezone_set('Asia/Jakarta');
    }

    $this->load->database();
    $this->load->helper(['url','form','security']);
    $this->load->library(['form_validation','session']);
    $this->load->model('Piket_daily_model', 'Piket');
    $this->load->model('Piket_signature_model', 'PSig');
}


    /* ===== Wrapper View ===== */
    private function render($view, $data = [])
    {
        $data['title'] = $data['title'] ?? 'Piket Daily';
        $this->load->view('templates/header',        $data);
        $this->load->view('templates/sidebar',       $data);
        $this->load->view('templates/content_start', $data);
        $this->load->view($view,                     $data);
        $this->load->view('templates/footer');
    }

    /* ===== List ===== */
   public function index()
{
    $page   = max(1, (int)$this->input->get('page'));
    $limit  = 25;
    $offset = ($page - 1) * $limit;

    $rows = $this->db->order_by('dibuat_pada', 'DESC')
                     ->limit($limit, $offset)
                     ->get('piket_daily')
                     ->result();

    // --- build matriks status sign per laporan ---
    $sign_matrix = [];

    if ($rows) {
        $ids = array_map(static function($r) { return (int)$r->id; }, $rows);

        // default semua false
        foreach ($ids as $rid) {
            $sign_matrix[$rid] = [
                'PIC'     => false,
                'CHECK'   => false,
                'CONFIRM' => false,
                'APPROVE' => false,
            ];
        }

        if (!empty($ids) && $this->db->table_exists('piket_daily_signatures')) {
            $q = $this->db->where_in('piket_id', $ids)
                          ->get('piket_daily_signatures');

            foreach ($q->result() as $sg) {
                $rid = (int)$sg->piket_id;

                // pakai role atau slot_role, lalu normalisasi
                $raw  = $sg->role ?? $sg->slot_role ?? '';
                $slot = strtoupper(trim((string)$raw));

                // kalau dulu pernah pakai "CONFIRMED"/"APPROVED", map ke slot baru
                if ($slot === 'CONFIRMED') $slot = 'CONFIRM';
                if ($slot === 'APPROVED')  $slot = 'APPROVE';

                if (isset($sign_matrix[$rid][$slot])) {
                    $sign_matrix[$rid][$slot] = true;
                }
            }
        }
    }

    $this->render('piket_daily/index', [
        'rows'        => $rows,
        'sign_matrix' => $sign_matrix,
        // rights_map sudah tidak dipakai, jadi tidak perlu dikirim
    ]);
}

    /* ===== Create Form ===== */
    public function create()
    {
        $this->render('piket_daily/create', [
            'title'        => 'Entry Piket Daily',
            'default_name' => (string) ($this->_session_name() ?: '')
        ]);
    }

    /* ===== Store ===== */
    public function store()
    {
        $this->_set_header_rules();
        if (!$this->form_validation->run()) {
            $this->_flash('error', strip_tags(validation_errors(' ',' ')));
            redirect('piket-daily/create');
            return;
        }

        $user_id      = $this->_session_uid();
        $tanggal      = $this->input->post('tanggal');
        $shift        = $this->input->post('shift');
        $lokasi       = $this->input->post('lokasi', true);
        $deskripsi    = $this->input->post('deskripsi', true);
        $nama_pengisi = $this->input->post('nama_pengisi', true);
        $jam_header   = trim($this->input->post('jam_header'));

        if (!$this->_valid_date($tanggal)) {
            $this->_flash('error','Tanggal tidak valid.');
            redirect('piket-daily/create');
            return;
        }

        $vj = $this->_validate_jam($tanggal, $jam_header);
        if ($vj !== true) {
            $this->_flash('error', $vj);
            redirect('piket-daily/create');
            return;
        }

        $secret   = getenv('APP_SIGNATURE_SECRET') ?: 'dhi_secret';
        $basis    = $tanggal.'|'.$shift.'|'.$lokasi.'|'.$user_id.'|'.date('c');
        $sig_hash = hash('sha256', $basis.$secret);

        $header = [
            'tanggal'        => $tanggal,
            'shift'          => $shift,
            'lokasi'         => $lokasi,
            'deskripsi'      => $deskripsi,
            'dibuat_oleh'    => $user_id,
            'signature_hash' => $sig_hash,
            'nama_pengisi'   => $nama_pengisi ?: ($this->_session_name() ?: null),
            'jam_header'     => $jam_header,
        ];

        $this->db->trans_start();
        $this->db->insert('piket_daily', $header);

        if ($this->db->affected_rows() !== 1) {
            $this->_flash('error','Insert header gagal: '.$this->_dberr());
            $this->db->trans_rollback();
            redirect('piket-daily/create');
            return;
        }

        $piket_id    = (int)$this->db->insert_id();
        $postedLines = $this->input->post('lines') ?: [];
        $urut        = 0;

        foreach ($postedLines as $i => $ln) {
            $issue = isset($ln['issue']) ? trim((string)$ln['issue']) : null;

            $row = [
                'piket_id'    => $piket_id,
                'section'     => isset($ln['section'])   ? trim((string)$ln['section'])   : null,
                'time_slot'   => isset($ln['time_slot']) ? trim((string)$ln['time_slot']) : null,
                'description' => null,
                'issue'       => $issue,
                'remarks'     => null,
                'is_issue'    => ($issue !== null && $issue !== '') ? 1 : 0,
                'urutan'      => $urut++,
            ];

            if ($this->db->field_exists('updated_by','piket_daily_lines')) {
                $row['updated_by'] = $user_id;
            }
            if ($this->db->field_exists('updated_at','piket_daily_lines')) {
                $row['updated_at'] = date('Y-m-d H:i:s');
            }

            $this->db->insert('piket_daily_lines', $row);
            if ($this->db->affected_rows() !== 1) {
                $this->_flash('error','Insert line gagal: '.$this->_dberr());
                $this->db->trans_rollback();
                redirect('piket-daily/create');
                return;
            }

            $this->_save_line_photos_with_compress($piket_id, (int)$i);
        }

        $this->db->trans_complete();
        if (!$this->db->trans_status()) {
            $this->_flash('error','Transaksi DB gagal: '.$this->_dberr());
            redirect('piket-daily/create');
            return;
        }

        $this->session->set_flashdata('piket_saved_id', $piket_id);
        $this->_flash('success','Piket Daily tersimpan.');
        redirect('piket-daily/create');
    }

    /* ===== Signature helpers & flow ===== */

    // status tombol (AJAX)
    public function signature_status($piket_id)
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $piket_id = (int)$piket_id;
        $map      = $this->_get_signature_map($piket_id);

        $status = [];
        foreach (self::SIG_ORDER as $slot) {
            $signed  = !empty($map[$slot]);
            $prev    = $this->_sig_prev_of($slot);
            $enabled = $signed ? false : ($prev ? !empty($map[$prev]) : true); // PIC default enabled
            $status[$slot] = ['signed'=>$signed, 'enabled'=>$enabled];
        }

        $this->_json(['ok'=>true,'status'=>$status]);
    }

    // Helper ambil row aman
    private function _db_row_safe(string $table, array $where)
    {
        if (!$this->db->table_exists($table)) {
            return null;
        }

        $q = $this->db->get_where($table, $where);
        return ($q && $q->num_rows()) ? $q->row() : null;
    }

    // PAD (canvas) tanda tangan
   public function sign_pad($piket_id, $slot_role = 'PIC')
{
    $piket_id  = (int)$piket_id;
    $slot_role = strtoupper($slot_role);
    $header    = $this->_get_header_or_404($piket_id);

    [$ok,$msg] = $this->_can_user_sign($piket_id, $slot_role);
    if (!$ok) {
        show_error($msg, 403);
        return;
    }

    $exists = $this->_db_row_safe('piket_daily_signatures', [
        'piket_id' => $piket_id,
        'role'     => $slot_role,
    ]);

    // <<– PENTING: pakai wrapper layout
    $this->render('piket_daily/sign_pad', [
        'header'    => $header,
        'slot_role' => $slot_role,
        'exists'    => $exists,
        'title'     => 'Tanda Tangan '.$slot_role,
    ]);
}

 
    // ===== Submit dari pad (POST) =====
public function sign_submit()
{
    // pastikan sudah login
    if (!$this->session->userdata('role')) {
        $this->session->set_userdata('after_login_redirect', current_url());
        redirect('login');
        return;
    }

    $piket_id  = (int)$this->input->post('piket_id');
    $slot_role = strtoupper(trim((string)$this->input->post('slot_role')));
    $note      = trim((string)$this->input->post('note'));

    // data dari form
    $dataUrl   = (string)$this->input->post('image_data');   // dari canvas
    $useStored = (bool)$this->input->post('use_stored');     // tombol "gunakan ttd tersimpan"
    $hasUpload = isset($_FILES['sign_file']) &&
                 (int)$_FILES['sign_file']['error'] === UPLOAD_ERR_OK;

    $allowed = ['PIC','CHECK','CONFIRM','APPROVE'];
    if (!$piket_id || !in_array($slot_role, $allowed, true)) {
        $this->_flash('error','Data tanda tangan tidak valid.');
        redirect('piket-daily');
        return;
    }

    // Pastikan report ada
    $this->_get_header_or_404($piket_id);

    // Cek urutan + hak
    [$ok,$msg] = $this->_can_user_sign($piket_id, $slot_role);
    if (!$ok) {
        $this->_flash('error',$msg);
        redirect('piket-daily');
        return;
    }

    // Cegah double sign  (cek role lowercase & uppercase + slot_role)
    $this->db->from('piket_daily_signatures');
    $this->db->where('piket_id', $piket_id);
    $this->db->group_start()
             ->where('role', strtolower($slot_role))
             ->or_where('role', strtoupper($slot_role));
    if ($this->db->field_exists('slot_role','piket_daily_signatures')) {
        $this->db->or_where('slot_role', $slot_role);
    }
    $this->db->group_end();
    $already = (int)$this->db->count_all_results();

    if ($already > 0) {
        $this->_flash('error','Slot '.$slot_role.' sudah ditandatangani.');
        redirect('piket-daily');
        return;
    }

    // ===== Tentukan MODE tanda tangan =====
    $mode = null;
    if ($useStored) {
        $mode = 'stored';
    } elseif ($hasUpload) {
        $mode = 'upload';
    } elseif (strpos($dataUrl, 'base64,') !== false) {
        $mode = 'canvas';
    }

    if ($mode === null) {
        $this->_flash('error','Tidak ada data tanda tangan yang dikirim.');
        redirect('piket-daily');
        return;
    }

    // folder simpan final
    $dirSign = rtrim(FCPATH,'/\\').DIRECTORY_SEPARATOR
             .'uploads'.DIRECTORY_SEPARATOR.'piket_signatures'.DIRECTORY_SEPARATOR;
    $this->_ensure_dir($dirSign);

    $relPath = null; // relative path ke DB

    // ====== MODE 1: GUNAKAN TTD TERSIMPAN ======
    if ($mode === 'stored') {

        $storedFull = $this->_locate_stored_signature();
        if (!$storedFull) {
            $this->_flash('error','Tanda tangan tersimpan untuk user ini tidak ditemukan.');
            redirect('piket-daily');
            return;
        }

        $ext = strtolower(pathinfo($storedFull, PATHINFO_EXTENSION));
        if (!in_array($ext, ['png','jpg','jpeg'], true)) {
            $this->_flash('error','Format tanda tangan tersimpan tidak didukung.');
            redirect('piket-daily');
            return;
        }

        $ext   = ($ext === 'jpeg') ? 'jpg' : $ext;
        $fname = 'SIG_'.$piket_id.'_'.$slot_role.'_'.time().'_'.bin2hex(random_bytes(3)).'.'.$ext;
        $dest  = $dirSign.$fname;

        if (!@copy($storedFull, $dest)) {
            $this->_flash('error','Gagal menyalin tanda tangan tersimpan.');
            redirect('piket-daily');
            return;
        }

        $relPath = 'uploads/piket_signatures/'.$fname;
    }

    // ====== MODE 2: UPLOAD FILE GAMBAR ======
    elseif ($mode === 'upload') {

        $tmp  = $_FILES['sign_file']['tmp_name'];
        $info = @getimagesize($tmp);

        if (!$info || !in_array($info['mime'], ['image/png','image/jpeg'], true)) {
            $this->_flash('error','File tanda tangan harus PNG atau JPG.');
            redirect('piket-daily');
            return;
        }

        $ext   = ($info['mime'] === 'image/png') ? 'png' : 'jpg';
        $fname = 'SIG_'.$piket_id.'_'.$slot_role.'_'.time().'_'.bin2hex(random_bytes(3)).'.'.$ext;
        $dest  = $dirSign.$fname;

        if (!@move_uploaded_file($tmp, $dest)) {
            $this->_flash('error','Gagal menyimpan file tanda tangan.');
            redirect('piket-daily');
            return;
        }

        $relPath = 'uploads/piket_signatures/'.$fname;
    }

    // ====== MODE 3: CANVAS (BASE64) ======
    elseif ($mode === 'canvas') {

        $dec = $this->_decode_dataurl($dataUrl);
        if (!$dec['ok']) {
            $this->_flash('error',$dec['msg']);
            redirect('piket-daily');
            return;
        }

        $ext   = ($dec['mime'] === 'image/jpeg') ? 'jpg' : 'png';
        $fname = 'SIG_'.$piket_id.'_'.$slot_role.'_'.time().'_'.bin2hex(random_bytes(3)).'.'.$ext;
        $full  = $dirSign.$fname;

        if (@file_put_contents($full, $dec['bin']) === false || !is_file($full)) {
            $this->_flash('error','Gagal menyimpan tanda tangan.');
            redirect('piket-daily');
            return;
        }

        $relPath = 'uploads/piket_signatures/'.$fname;
    }

    if (!$relPath) {
        $this->_flash('error','Gagal menentukan path tanda tangan.');
        redirect('piket-daily');
        return;
    }

    // Insert DB
    $uid  = (int)($this->session->userdata('user_id')
            ?: $this->session->userdata('id') ?: 0);
    $name = (string)($this->session->userdata('nama')
            ?: $this->session->userdata('username') ?: '');

    $this->db->insert('piket_daily_signatures', [
        'piket_id'    => $piket_id,
        'role'        => strtolower($slot_role),   // enum di DB: pic/check/confirmed/approval
        'user_id'     => $uid,
        'signer_name' => $name,
        'note'        => ($note !== '' ? $note : null),
        'image_path'  => $relPath,
        'signed_at'   => date('Y-m-d H:i:s'),
        'ip_address'  => (string)$this->input->ip_address(),
        'user_agent'  => substr((string)$this->input->user_agent(), 0, 255),
    ]);

    if ($this->db->affected_rows() !== 1) {
        $this->_flash('error','Simpan tanda tangan gagal.');
        redirect('piket-daily');
        return;
    }

    // kirim notif ke role berikutnya
    $this->_notify_next_role($piket_id, $slot_role);

    $this->_flash('success','Tanda tangan '.$slot_role.' tersimpan (mode: '.$mode.').');
    redirect('piket-daily');
}


    /** Decoder dataURL kecil & aman (png/jpeg) */
    private function _decode_dataurl(string $dataUrl): array
    {
        // data:image/png;base64,xxxx  | data:image/jpeg;base64,xxxx
        if (!preg_match('#^data:(image/(png|jpeg));base64,(.+)$#i', $dataUrl, $m)) {
            return ['ok'=>false,'msg'=>'Format gambar tidak didukung.'];
        }

        $mime = strtolower($m[1]);
        $b64  = $m[3];

        // batasi ~2MB string base64
        if (strlen($b64) > 2_000_000) {
            return ['ok'=>false,'msg'=>'Ukuran gambar terlalu besar.'];
        }

        $bin = base64_decode($b64, true);
        if ($bin === false || $bin === '') {
            return ['ok'=>false,'msg'=>'Gagal decode gambar.'];
        }

        // verifikasi cepat signature file
        if (function_exists('finfo_buffer')) {
            $f  = new finfo(FILEINFO_MIME_TYPE);
            $mm = $f->buffer($bin) ?: '';
            if (!in_array($mm, ['image/png','image/jpeg'], true)) {
                return ['ok'=>false,'msg'=>'Mime file tidak valid.'];
            }
            $mime = $mm;
        }

        return ['ok'=>true,'mime'=>$mime,'bin'=>$bin];
    }

    /**
 * Cari file tanda tangan master berdasarkan username.
 * Lokasi: uploads/piket_signatures_data/{username}.png/jpg/jpeg
 * Return: fullpath atau null jika tidak ada.
 */
private function _locate_stored_signature(): ?string
{
    $username = (string)$this->session->userdata('username');
    if ($username === '') {
        return null;
    }

    $baseDir = rtrim(FCPATH,'/\\').DIRECTORY_SEPARATOR
             .'uploads'.DIRECTORY_SEPARATOR
             .'piket_signatures_data'.DIRECTORY_SEPARATOR;

    foreach (['png','jpg','jpeg'] as $ext) {
        $full = $baseDir.$username.'.'.$ext;
        if (is_file($full)) {
            return $full;
        }
    }
    return null;
}


    /* ===== Edit ===== */
    public function edit($id)
    {
        $id     = (int)$id;
        $header = $this->_get_header_or_404($id);
        $this->_require_owner($header);

        // seed default row jika belum ada
        if ($this->db->table_exists('piket_daily_lines') &&
            !$this->db->where('piket_id',$header->id)->count_all_results('piket_daily_lines')) {

            $defaults = [
                ['section'=>'PRODUCTION (Vacuum, Press, DHC)','time_slot'=>null,'urutan'=>10],
                ['section'=>'DELIVERY',                         'time_slot'=>null,'urutan'=>20],
                ['section'=>'SAFETY',                           'time_slot'=>null,'urutan'=>30],
                ['section'=>'CHECK POINT KONDISI PINTU ANTISIPASI SERANGGA','time_slot'=>'Jam 17.00 - 19.00 wib','urutan'=>40],
                ['section'=>null,                               'time_slot'=>'Jam 19.00 - 20.00 wib','urutan'=>41],
                ['section'=>null,                               'time_slot'=>'Jam 20.00 wib - dst','urutan'=>42],
                ['section'=>'5-S',                              'time_slot'=>null,'urutan'=>50],
                ['section'=>'SAVING ENERGY (A/C, Lamp, M/C)',   'time_slot'=>null,'urutan'=>60],
                ['section'=>'ABSENSI & AKTIVITAS KARYAWAN',     'time_slot'=>null,'urutan'=>70],
            ];
            foreach ($defaults as $d) {
                $d['piket_id'] = $header->id;
                $this->db->insert('piket_daily_lines', $d);
            }
        }

        $lines = $this->db->order_by('urutan ASC, id ASC')
                          ->get_where('piket_daily_lines', ['piket_id'=>$header->id])
                          ->result();

        $photos_by_line = [];
        if ($this->db->table_exists('piket_daily_photos')) {
            $q = $this->db->order_by('urutan ASC, id ASC')
                          ->get_where('piket_daily_photos', ['piket_id'=>$header->id]);
            $hasLineIndex = $this->db->field_exists('line_index','piket_daily_photos');

            foreach ($q->result() as $p) {
                $idx = $hasLineIndex ? (string)($p->line_index ?? '') : '';
                $photos_by_line[$idx][] = $p;
            }
        }

        $this->render('piket_daily/editor', [
            'title'          => 'Edit Laporan Piket',
            'header'         => $header,
            'lines'          => $lines,
            'photos_by_line' => $photos_by_line,
        ]);
    }

    /* ===== Update header ===== */
    public function update_header($id)
    {
        $id  = (int)$id;
        $row = $this->db->get_where('piket_daily', ['id'=>$id])->row();
        if (!$row) {
            show_404();
        }
        $this->_require_owner($row);

        $this->_set_header_rules();
        if (!$this->form_validation->run()) {
            $this->_flash('error', strip_tags(validation_errors(' ',' ')));
            redirect('piket-daily/edit/'.$id);
            return;
        }

        $upd = [
            'nama_pengisi' => $this->input->post('nama_pengisi', true),
            'tanggal'      => $this->input->post('tanggal'),
            'jam_header'   => $this->input->post('jam_header', true),
        ];

        $msgJam = $this->_validate_jam($upd['tanggal'], $upd['jam_header']);
        if ($msgJam !== true) {
            $this->_flash('error', $msgJam);
            redirect('piket-daily/edit/'.$id);
            return;
        }

        $this->db->where('id',$id)->update('piket_daily', $upd);
        $this->_flash('success','Header tersimpan.');
        redirect('piket-daily/edit/'.$id);
    }

    /* ===== Inline save cell (AJAX) ===== */
    public function save_line()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $line_id = (int)$this->input->post('id');
        $field   = $this->input->post('field', true);
        $value   = $this->input->post('value', false);

        if (!$line_id || !in_array($field, ['issue','remarks','description'], true)) {
            $this->_json(['ok'=>false,'msg'=>'Bad request'], 400);
            return;
        }

        $line = $this->db->get_where('piket_daily_lines',['id'=>$line_id])->row();
        if (!$line) {
            $this->_json(['ok'=>false],404);
            return;
        }

        $header = $this->_get_header_or_404($line->piket_id);
        $this->_require_owner($header, true);

        $clean = trim(strip_tags($value, ''));
        $ok    = $this->db->where('id', $line_id)->update('piket_daily_lines', [
            $field       => $clean,
            'is_issue'   => ($field==='issue' && $clean!=='') ? 1 : (int)$line->is_issue,
            'updated_by' => $this->_session_uid(),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);

        $this->_json(['ok'=>$ok]);
    }

    /* ===== Save satu baris issue (AJAX) ===== */
    public function save_row()
    {
        if (!$this->input->is_ajax_request()) {
            $this->output->set_status_header(400);
        }

        $line_id = (int)$this->input->post('id');
        $issue   = (string)$this->input->post('issue', false);
        if (!$line_id) {
            return $this->_json(['ok'=>false,'msg'=>'Bad request'], 400);
        }

        $line = $this->db->get_where('piket_daily_lines',['id'=>$line_id])->row();
        if (!$line) {
            return $this->_json(['ok'=>false,'msg'=>'Not found'],404);
        }

        $header = $this->_get_header_or_404($line->piket_id);
        if (!$this->_can_edit($header)) {
            return $this->_json(['ok'=>false,'msg'=>'Forbidden'],403);
        }

        $clean = trim(strip_tags($issue,''));
        $ok    = $this->db->where('id',$line_id)->update('piket_daily_lines',[
            'issue'      => $clean,
            'is_issue'   => ($clean!=='' ? 1 : 0),
            'updated_by' => $this->_session_uid(),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);

        return $this->_json(['ok'=>$ok]);
    }

    /* ===== Upload satu foto (AJAX) ===== */
    public function upload_one_photo()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $piket_id  = (int)$this->input->post('piket_id');
        $lineIndex = (int)$this->input->post('line_index');
        $slot      = (int)$this->input->post('slot'); // 1..3

        if (!$piket_id || $slot < 1 || $slot > 3 || !isset($_FILES['photo'])) {
            return $this->_json(['ok'=>false,'msg'=>'Bad request'], 400);
        }

        $hasLineIndex = $this->db->field_exists('line_index','piket_daily_photos');
        $qb = $this->db->where('piket_id', $piket_id);
        if ($hasLineIndex) {
            $qb->where('line_index', $lineIndex);
        }
        $count = (int) $qb->count_all_results('piket_daily_photos');
        if ($count >= 3) {
            return $this->_json(['ok'=>false,'msg'=>'Slot foto sudah penuh (3).'], 200);
        }

        $this->_ensure_dir(FCPATH.'uploads/piket_daily/');
        if ((int)$_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
            return $this->_json(['ok'=>false,'msg'=>'Upload error.'], 200);
        }

        $tmp      = $_FILES['photo']['tmp_name'];
        $filename = 'PD_'.time().'_'.$lineIndex.'_'.($slot-1).'_'.bin2hex(random_bytes(3)).'.jpg';
        $dest     = FCPATH.'uploads/piket_daily/'.$filename;

        if (!$this->_compress_to_1mb($tmp, $dest, 1024, 1920)) {
            return $this->_json(['ok'=>false,'msg'=>'Kompresi gagal.'], 200);
        }

        $data = [
            'piket_id'  => $piket_id,
            'file_path' => 'uploads/piket_daily/'.$filename,
            'caption'   => null,
            'urutan'    => $slot-1
        ];
        if ($hasLineIndex) {
            $data['line_index'] = $lineIndex;
        }

        $find = $this->db->where('piket_id',$piket_id);
        if ($hasLineIndex) {
            $find->where('line_index',$lineIndex);
        }
        $find->where('urutan', $slot-1);
        $exists = $find->get('piket_daily_photos')->row();

        if ($exists) {
            $old = FCPATH.$exists->file_path;
            if (is_file($old)) {
                @unlink($old);
            }
            $this->db->where('id',$exists->id)->update('piket_daily_photos', $data);
        } else {
            $this->db->insert('piket_daily_photos', $data);
        }

        $qb2 = $this->db->where('piket_id',$piket_id);
        if ($hasLineIndex) {
            $qb2->where('line_index',$lineIndex);
        }
        $now       = (int) $qb2->count_all_results('piket_daily_photos');
        $remaining = max(0, 3 - $now);

        return $this->_json([
            'ok'        => true,
            'slot'      => $slot,
            'path'      => base_url($data['file_path']),
            'remaining' => $remaining
        ]);
    }

    /* ===== Hapus foto (AJAX) ===== */
    public function delete_photo()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $id = (int)$this->input->post('photo_id');
        if (!$id) {
            $this->_json(['ok'=>false,'msg'=>'Bad request'],400);
            return;
        }

        $p = $this->db->get_where('piket_daily_photos', ['id'=>$id])->row();
        if (!$p) {
            $this->_json(['ok'=>true]);
            return;
        }

        $header = $this->_get_header_or_404($p->piket_id);
        $this->_require_owner($header, true);

        $this->db->delete('piket_daily_photos', ['id'=>$id]);
        $path = FCPATH.$p->file_path;
        if (is_file($path)) {
            @unlink($path);
        }

        $this->_json(['ok'=>true]);
    }

    /* ===== Print A4 ===== */
   public function print($id)
{
    // --- HEADER LAPORAN ---
    $header = $this->_get_header_or_404((int)$id);

    // --- LINES ---
    $lines = $this->db->order_by('urutan ASC, id ASC')
                      ->get_where('piket_daily_lines', ['piket_id' => $header->id])
                      ->result();

    // --- PHOTOS BY LINE ---
    $photos_by_line = [];
    $hasLineIndex   = $this->db->field_exists('line_index', 'piket_daily_photos');

    $ph = $this->db->order_by('line_index ASC, urutan ASC, id ASC')
                   ->get_where('piket_daily_photos', ['piket_id' => $header->id])
                   ->result();

    foreach ($ph as $p) {
        $idx = ($hasLineIndex && $p->line_index !== null && $p->line_index !== '')
             ? (string) max(0, (int)$p->line_index)
             : '0';
        $photos_by_line[$idx][] = $p;
    }

    // --- SIGNATURES (LANGSUNG QUERY KE TABEL) ---
    $rows = $this->db
        ->order_by('id', 'ASC')
        ->get_where('piket_daily_signatures', ['piket_id' => (int)$header->id])
        ->result();

    // siapkan slot default
    $Sign = [
        'PIC'      => null,
        'CHECK'    => null,
        'CONFIRM'  => null,
        'APPROVE'  => null,
    ];

    foreach ($rows as $r) {
        // role bisa di kolom 'role' (baru) atau 'slot_role' (lama), dan bisa lowercase
        $roleVal = '';
        if (!empty($r->role)) {
            $roleVal = $r->role;
        } elseif (!empty($r->slot_role)) {
            $roleVal = $r->slot_role;
        }

        $key = strtoupper(trim((string)$roleVal));   // 'pic' -> 'PIC'
        if (!isset($Sign[$key])) {
            continue; // kalau bukan PIC/CHECK/CONFIRM/APPROVE, skip
        }

        // alias nama kalau kolom beda-beda
        if (empty($r->signer_name) && !empty($r->signed_by_name)) {
            $r->signer_name = $r->signed_by_name;
        }

        // build src gambar (data URI kalau file ada, fallback base_url)
        $r->img_src = '';
        if (!empty($r->image_path)) {
            $r->img_src = $this->_sig_img_src($r->image_path);
        }

        $Sign[$key] = $r;
    }

    // --- BUILD NOTES DARI TABEL SIGNATURE ---
    $Notes    = [];
    $labelMap = [
        'CHECK'   => 'Checker',
        'CONFIRM' => 'Confirmed',
        'APPROVE' => 'Approval',
    ];

    foreach ($rows as $r) {
        $roleVal = '';
        if (!empty($r->role)) {
            $roleVal = $r->role;
        } elseif (!empty($r->slot_role)) {
            $roleVal = $r->slot_role;
        }

        $key = strtoupper(trim((string)$roleVal));
        if (!isset($labelMap[$key])) {
            continue; // cuma kumpulin note untuk CHECK / CONFIRM / APPROVE
        }

        // kalau kolom note kosong, skip
        if (!isset($r->note) || trim((string)$r->note) === '') {
            continue;
        }

        // nama
        $name = '';
        if (!empty($r->signer_name)) {
            $name = $r->signer_name;
        } elseif (!empty($r->signed_by_name)) {
            $name = $r->signed_by_name;
        }

        // waktu
        $time = '';
        if (!empty($r->signed_at)) {
            $ts   = strtotime($r->signed_at);
            $time = $ts ? date('d-m-Y H:i', $ts) : $r->signed_at;
        }

        $Notes[] = [
            'role' => $labelMap[$key],
            'name' => $name,
            'time' => $time,
            'text' => (string)$r->note,
        ];
    }

    // --- RENDER VIEW KE HTML ---
    $html = $this->load->view('piket_daily/print_a4', [
        'header'         => $header,
        'lines'          => $lines,
        'photos_by_line' => $photos_by_line,
        'Sign'           => $Sign,   // penting: S BESAR, sama kayak di view
        'Notes'          => $Notes,  // <<-- kirim notes ke view
    ], true);

    // --- GENERATE PDF ---
    $this->load->library('pdfgenerator');

    if (property_exists($this->pdfgenerator, 'dompdf') && $this->pdfgenerator->dompdf) {
        $this->pdfgenerator->dompdf->set_option('isRemoteEnabled', true);
    } elseif (method_exists($this->pdfgenerator, 'set_option')) {
        $this->pdfgenerator->set_option('isRemoteEnabled', true);
    }

    $fname = 'PiketDaily_' . $header->tanggal . '_' . ($header->nama_pengisi ?: 'PIC');
    $this->pdfgenerator->generate($html, $fname, 'A4', 'portrait');
}


    /**
     * Helper: jadikan path image jadi src absolut untuk view/PDF.
     * Jika file ada -> data URI, else fallback base_url.
     */
    private function _sig_img_src(string $image_path): string
    {
        $rel  = ltrim($image_path, '/\\');
        $full = rtrim(FCPATH,'/\\').DIRECTORY_SEPARATOR.$rel;

        if (is_file($full) && is_readable($full)) {
            $ext  = strtolower(pathinfo($full, PATHINFO_EXTENSION));
            $mime = ($ext === 'png') ? 'image/png' : 'image/jpeg';
            $bin  = @file_get_contents($full);
            if ($bin !== false && $bin !== '') {
                return 'data:'.$mime.';base64,'.base64_encode($bin);
            }
        }

        return base_url($rel);
    }

    /* ===== Delete report ===== */
    public function destroy($id)
    {
        $id     = (int)$id;
        $header = $this->_get_header_or_404($id);
        $this->_require_owner($header);

        $this->db->trans_start();

        $photos = $this->db->get_where('piket_daily_photos', ['piket_id'=>$id])->result();
        foreach ($photos as $p) {
            $path = FCPATH.$p->file_path;
            if (is_file($path)) {
                @unlink($path);
            }
        }
        $this->db->delete('piket_daily_photos', ['piket_id'=>$id]);

        if ($this->db->table_exists('piket_daily_signatures')) {
            $sigs = $this->db->get_where('piket_daily_signatures', ['piket_id'=>$id])->result();
            foreach ($sigs as $s) {
                if (!empty($s->image_path)) {
                    $sp = FCPATH.$s->image_path;
                    if (is_file($sp)) {
                        @unlink($sp);
                    }
                }
            }
            $this->db->delete('piket_daily_signatures', ['piket_id'=>$id]);
        }

        if ($this->db->table_exists('piket_daily_lines')) {
            $this->db->delete('piket_daily_lines', ['piket_id'=>$id]);
        }
        $this->db->delete('piket_daily', ['id'=>$id]);

        $this->db->trans_complete();

        $this->_flash('success','Data dihapus.');
        redirect('piket-daily');
    }

    /* ===== Helpers ===== */

    private function _json($arr, $code = 200)
    {
        $this->output->set_content_type('application/json')
                     ->set_status_header($code)
                     ->set_output(json_encode($arr));
    }

    private function _set_header_rules()
    {
        $this->form_validation->set_rules('nama_pengisi','Nama','required|trim');
        $this->form_validation->set_rules('tanggal','Tanggal','required');
        $this->form_validation->set_rules('jam_header','Jam','required|trim');
    }

    private function _valid_date($ymd)
    {
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $ymd)) {
            return false;
        }
        [$Y,$m,$d] = array_map('intval', explode('-', $ymd));
        return checkdate($m,$d,$Y);
    }

    private function _validate_jam($tanggal, $jam_header)
    {
        $dayIndex   = (int)date('w', strtotime($tanggal));
        $isWeekday  = in_array($dayIndex,[1,2,3,4,5],true);
        $isSaturday = ($dayIndex===6);
        $isSunday   = ($dayIndex===0);

        if ($isSunday) {
            return 'Entri hari Minggu belum diizinkan. Pilih Senin–Sabtu.';
        }
        if (!preg_match(self::JAM_REGEX, $jam_header)) {
            return 'Format jam harus "HH.MM - HH.MM".';
        }
        if ($isWeekday  && $jam_header !== self::WEEKDAY_JAM) {
            return 'Untuk Senin–Jumat jam wajib "'.self::WEEKDAY_JAM.'".';
        }
        if ($isSaturday && $jam_header !== self::SATURDAY_JAM) {
            return 'Untuk Sabtu jam wajib "'.self::SATURDAY_JAM.'".';
        }
        return true;
    }

    private function _save_line_photos_with_compress($piket_id, $lineIndex)
    {
        $saveDir = FCPATH.'uploads/piket_daily/';
        $this->_ensure_dir($saveDir);

        $hasLineIndex = $this->db->field_exists('line_index','piket_daily_photos');
        $added        = 0;

        // mode baru: line_photo_{index}_{slot}
        for ($slot = 1; $slot <= 3; $slot++) {
            $field = "line_photo_{$lineIndex}_{$slot}";
            if (!isset($_FILES[$field]) || empty($_FILES[$field]['name'])) {
                continue;
            }
            if ((int)$_FILES[$field]['error'] !== UPLOAD_ERR_OK) {
                continue;
            }

            $tmp      = $_FILES[$field]['tmp_name'];
            $filename = 'PD_'.time().'_'.$lineIndex.'_'.($slot-1).'_'.bin2hex(random_bytes(3)).'.jpg';
            $dest     = $saveDir.$filename;

            if ($this->_compress_to_1mb($tmp, $dest, 1024, 1920)) {
                $ins = [
                    'piket_id'  => $piket_id,
                    'file_path' => 'uploads/piket_daily/'.$filename,
                    'caption'   => null,
                    'urutan'    => $slot-1,
                ];
                if ($hasLineIndex) {
                    $ins['line_index'] = $lineIndex;
                }
                $this->db->insert('piket_daily_photos', $ins);
                $added++;
            }
        }

        if ($added > 0) {
            return;
        }

        // mode lama: multiple upload line_photos_{index}[]
        $oldField = "line_photos_{$lineIndex}";
        if (!isset($_FILES[$oldField]['name'][0]) || $_FILES[$oldField]['name'][0] === '') {
            return;
        }

        $arr   = $_FILES[$oldField];
        $count = is_array($arr['name']) ? min(count($arr['name']), 3) : 0;

        for ($k=0; $k<$count; $k++) {
            if ((int)$arr['error'][$k] !== UPLOAD_ERR_OK) {
                continue;
            }

            $tmp      = $arr['tmp_name'][$k];
            $filename = 'PD_'.time().'_'.$lineIndex.'_'.$k.'_'.bin2hex(random_bytes(4)).'.jpg';
            $dest     = $saveDir.$filename;

            if ($this->_compress_to_1mb($tmp, $dest, 1024, 1920)) {
                $ins = [
                    'piket_id'  => $piket_id,
                    'file_path' => 'uploads/piket_daily/'.$filename,
                    'caption'   => null,
                    'urutan'    => $k,
                ];
                if ($hasLineIndex) {
                    $ins['line_index'] = $lineIndex;
                }
                $this->db->insert('piket_daily_photos', $ins);
            }
        }
    }

    private function _flash($type, $msg)
    {
        $key = $type === 'success' ? 'piket_success' : 'piket_error';
        $this->session->set_flashdata($key, $msg);
    }

    private function _ensure_dir($path)
    {
        if (!is_dir($path)) {
            @mkdir($path, 0775, true);
        }
    }

    private function _compress_to_1mb($srcTmpPath, $destFullPath, $targetKB = 1024, $maxWidth = 1920)
    {
        if (!is_file($srcTmpPath)) {
            return false;
        }

        $info = @getimagesize($srcTmpPath);
        if (!$info) {
            $img = @imagecreatefromstring(@file_get_contents($srcTmpPath));
        } else {
            switch ($info['mime']) {
                case 'image/png':
                    $img = imagecreatefrompng($srcTmpPath);
                    break;
                case 'image/jpeg':
                    $img = imagecreatefromjpeg($srcTmpPath);
                    break;
                default:
                    $img = @imagecreatefromstring(@file_get_contents($srcTmpPath));
                    break;
            }
        }

        if (!$img) {
            return false;
        }

        // orientasi
        if (function_exists('exif_read_data') && (!isset($info['mime']) || $info['mime'] === 'image/jpeg')) {
            $exif = @exif_read_data($srcTmpPath);
            if (!empty($exif['Orientation'])) {
                switch ($exif['Orientation']) {
                    case 3: $img = imagerotate($img, 180, 0); break;
                    case 6: $img = imagerotate($img, -90, 0); break;
                    case 8: $img = imagerotate($img,  90, 0); break;
                }
            }
        }

        $w = imagesx($img);
        $h = imagesy($img);

        // rotate portrait
        if ($h > $w) {
            $img = imagerotate($img, -90, 0);
            $w   = imagesx($img);
            $h   = imagesy($img);
        }

        // resize jika lebih besar dari max width
        if ($w > $maxWidth) {
            $newW = $maxWidth;
            $newH = (int) round($h * ($newW / $w));
            $tmp  = imagecreatetruecolor($newW, $newH);
            imagecopyresampled($tmp, $img, 0, 0, 0, 0, $newW, $newH, $w, $h);
            imagedestroy($img);
            $img = $tmp;
        }

        // compress ke target size
        $quality = 90;
        do {
            ob_start();
            imagejpeg($img, null, $quality);
            $data   = ob_get_clean();
            $sizeKB = strlen($data) / 1024;
            $quality -= 5;
        } while ($sizeKB > $targetKB && $quality >= 40);

        imagedestroy($img);

        if (empty($data)) {
            return false;
        }

        if (file_put_contents($destFullPath, $data) === false) {
            return false;
        }

        return is_file($destFullPath) && filesize($destFullPath) > 0;
    }

    private function _dberr()
    {
        $err = $this->db->error();
        if (!empty($err['message'])) {
            return $err['message'];
        }
        if (method_exists($this->db, 'last_query')) {
            return 'DB error. Last query: '.$this->db->last_query();
        }
        return 'DB error (unknown).';
    }

    private function _get_header_or_404($id)
    {
        $row = $this->db->get_where('piket_daily', ['id'=>$id])->row();
        if (!$row) {
            show_404();
        }
        return $row;
    }

    private function _can_edit($header)
    {
        $uid  = $this->_session_uid();
        $role = $this->_session_role();
        return ($uid && (int)$header->dibuat_oleh === $uid) || in_array($role, ['admin','superadmin'], true);
    }

    private function _require_owner($header, $as_json = false)
    {
        if ($this->_can_edit($header)) {
            return;
        }
        if ($as_json) {
            $this->_json(['ok'=>false,'msg'=>'Forbidden'], 403);
            exit;
        }
        show_error('Forbidden', 403);
    }

    private function _report_payload_hash($piket_id)
    {
        $h = $this->db->get_where('piket_daily', ['id'=>$piket_id])->row_array();
        $l = $this->db->order_by('urutan,id')->get_where('piket_daily_lines',  ['piket_id'=>$piket_id])->result_array();
        $p = $this->db->order_by('urutan,id')->get_where('piket_daily_photos', ['piket_id'=>$piket_id])->result_array();

        $secret = getenv('APP_SIGNATURE_SECRET') ?: 'dhi_secret';
        $raw = json_encode([
            'header' => [
                'tanggal' => $h['tanggal'] ?? null,
                'nama'    => $h['nama_pengisi'] ?? null,
                'jam'     => $h['jam_header'] ?? null,
            ],
            'lines'  => array_map(static function($x){
                return [$x['section'],$x['time_slot'],$x['issue'],$x['urutan']];
            }, $l),
            'photos' => array_map(static function($x){
                return [$x['line_index'] ?? null,$x['file_path'],$x['urutan']];
            }, $p),
        ], JSON_UNESCAPED_UNICODE);

        return hash('sha256', $raw.$secret);
    }

    private function _sig_prev_of($slot)
    {
        $idx = array_search($slot, self::SIG_ORDER, true);
        return ($idx > 0) ? self::SIG_ORDER[$idx-1] : null;
    }

    private function _get_signature_map($piket_id)
{
    $rows = $this->db
        ->order_by('id','ASC')
        ->get_where('piket_daily_signatures', ['piket_id' => (int)$piket_id])
        ->result();

    $map = ['PIC'=>null,'CHECK'=>null,'CONFIRM'=>null,'APPROVE'=>null];

    foreach ($rows as $r) {
        // pakai role atau slot_role (biar kompatibel semua skema)
        $key = strtoupper((string)($r->role ?? $r->slot_role ?? ''));
        if (!isset($map[$key])) {
            continue;
        }

        if (empty($r->signer_name) && !empty($r->signed_by_name)) {
            $r->signer_name = $r->signed_by_name;
        }

        if (!empty($r->image_path)) {
            $r->img_src = $this->_sig_img_src($r->image_path);
        }

        $map[$key] = $r;
    }

    return $map;
}


   private function _can_user_sign($piket_id, $slot_role)
{
    $piket_id  = (int)$piket_id;
    $slot_role = strtoupper(trim((string)$slot_role));

    $uid  = $this->_session_uid();
    $role = $this->_session_role(); // admin/user/check/confirmed/approval (lowercase)

    if (!$uid) {
        return [false, 'Anda belum login.'];
    }

    /* ============================================================
     * 1. CEK TAHAP SEBELUMNYA (PIC -> CHECK -> CONFIRM -> APPROVE)
     *    Langsung query ke piket_daily_signatures, tapi aman
     *    terhadap struktur lama/baru.
     * ============================================================ */
    if ($slot_role !== 'PIC') { // PIC tidak punya tahap sebelumnya

        $prev = $this->_sig_prev_of($slot_role);   // CHECK->PIC, CONFIRM->CHECK, APPROVE->CONFIRM

        if ($prev) {

            if (!$this->db->table_exists('piket_daily_signatures')) {
                return [false, 'Tabel tanda tangan belum tersedia.'];
            }

            $this->db->from('piket_daily_signatures');
            $this->db->where('piket_id', $piket_id);

            $this->db->group_start();
            // role di DB biasanya enum: 'pic','check','confirmed','approval'
            $this->db->where('role', strtolower($prev))
                     ->or_where('role', strtoupper($prev));
            // kalau ada kolom slot_role (skema lama), ikut dipakai
            if ($this->db->field_exists('slot_role', 'piket_daily_signatures')) {
                $this->db->or_where('slot_role', $prev);
            }
            $this->db->group_end();

            $count_prev = (int) $this->db->count_all_results();

            if ($count_prev === 0) {
                return [false, 'Tahap sebelumnya ('.$prev.') belum ditandatangani.'];
            }
        }
    }

    /* ============================================================
     * 2. CEK HAK USER UNTUK SLOT YANG DIMINTA
     * ============================================================ */

    switch ($slot_role) {

        /* ---------- SLOT PIC ---------- */
        case 'PIC':
            // boleh PIC kalau:
            //  - admin, atau
            //  - pembuat laporan, atau
            //  - nama session sama dengan nama_pengisi
            $h = $this->_get_header_or_404($piket_id);

            $sessionName = $this->_session_name();
            $isOwner     = ($uid && (int)$h->dibuat_oleh === $uid);
            $nameMatch   = ($sessionName && strcasecmp($sessionName, (string)$h->nama_pengisi) === 0);
            $isAdmin     = ($role === 'admin');

            if ($isAdmin || $isOwner || $nameMatch) {
                return [true, null];
            }
            return [false, 'Anda tidak berhak menandatangani slot PIC.'];

        /* ---------- SLOT CHECK ---------- */
        case 'CHECK':
            if (in_array($role, ['check','admin'], true)) {
                return [true, null];
            }
            return [false, 'Anda tidak berhak menandatangani slot CHECK.'];

        /* ---------- SLOT CONFIRM ---------- */
        case 'CONFIRM':
            if (in_array($role, ['confirmed','admin'], true)) {
                return [true, null];
            }
            return [false, 'Anda tidak berhak menandatangani slot CONFIRM.'];

        /* ---------- SLOT APPROVE ---------- */
        case 'APPROVE':
            if (in_array($role, ['approval','admin'], true)) {
                return [true, null];
            }
            return [false, 'Anda tidak berhak menandatangani slot APPROVE.'];

        default:
            return [false, 'Slot tanda tangan tidak dikenal.'];
    }
}


private function _notify_next_role($piket_id, $slot_role_done)
{
    // normalisasi slot yang barusan selesai
    $slot_role_done = strtoupper((string)$slot_role_done);

    // Cari slot berikutnya (PIC -> CHECK -> CONFIRM -> APPROVE)
    $nextSlot = $this->_sig_next_of($slot_role_done);
    if (!$nextSlot) {
        // APPROVE sudah akhir, tidak ada notif lagi
        return;
    }

    // Terjemahkan slot ke tb_user.role (check / confirmed / approval)
    $roleToNotify = $this->_user_role_for_slot($nextSlot);
    if (!$roleToNotify) {
        return;
    }

    // Ambil header buat info di email
    $header = $this->_get_header_or_404($piket_id);

    // Cari semua user dengan role tersebut
    $users = $this->db->select('email, username, nama')
                      ->from('tb_user')
                      ->where('role', $roleToNotify)
                      ->get()
                      ->result();

    if (!$users) {
        return;
    }

    // ====== LINK YANG BENAR ======
    // controller: PiketDaily
    // method   : sign_pad($piket_id, $slot_role)
    // Link ke tanda tangan & PDF
    $linkPad  = site_url('PiketDaily/sign_pad/'.$piket_id.'/'.$nextSlot);
    $linkView = site_url('piket-daily/print/'.$piket_id);

    $subject = "[Piket Daily] Menunggu tanda tangan {$nextSlot}";

    // --- BODY EMAIL HTML, dibuat rapi ---
    $tanggal  = htmlspecialchars($header->tanggal, ENT_QUOTES, 'UTF-8');
    $picNama  = htmlspecialchars($header->nama_pengisi ?: '-', ENT_QUOTES, 'UTF-8');
    $slotText = htmlspecialchars($nextSlot, ENT_QUOTES, 'UTF-8');

    $message =
        "<p>Halo,</p>"
      . "<p>Laporan <strong>Piket Daily</strong> berikut memerlukan tanda tangan "
      . "<strong>{$slotText}</strong>:</p>"
      . "<table cellpadding=\"2\" cellspacing=\"0\" style=\"border-collapse:collapse;\">"
      . "  <tr>"
      . "    <td style=\"width:80px;\">Tanggal</td><td>: {$tanggal}</td>"
      . "  </tr>"
      . "  <tr>"
      . "    <td>PIC</td><td>: {$picNama}</td>"
      . "  </tr>"
      . "</table>"
      . "<p>"
      . "Tanda tangan di sini:<br>"
      . "<a href=\"{$linkPad}\">{$linkPad}</a>"
      . "</p>"
      . "<p>"
      . "Lihat laporan (PDF):<br>"
      . "<a href=\"{$linkView}\">{$linkView}</a>"
      . "</p>"
      . "<p>Terima kasih.</p>";

    $this->load->library('email');

    foreach ($users as $u) {
        if (empty($u->email)) {
            continue;
        }

        $this->email->clear();
        $this->email->from('no-reply@yourapp.local', 'Piket Daily');
        $this->email->to($u->email);
        $this->email->subject($subject);
        $this->email->message($message);

        // jangan bikin fatal kalau gagal kirim
        @$this->email->send();
    }
}


    private function _sig_next_of($slot)
    {
        $idx = array_search($slot, self::SIG_ORDER, true);
        return ($idx !== false && isset(self::SIG_ORDER[$idx+1])) ? self::SIG_ORDER[$idx+1] : null;
    }

    private function _user_role_for_slot(string $slot): ?string
    {
    switch (strtoupper($slot)) {
        case 'PIC':     return 'user';       // tidak dipakai untuk notif, tapi biar lengkap
        case 'CHECK':   return 'check';
        case 'CONFIRM': return 'confirmed';
        case 'APPROVE': return 'approval';
        default:        return null;
    }
    }


    /* ===== Session helpers ===== */
    private function _session_uid()
    {
        return (int)(
            $this->session->userdata('user_id')
            ?: $this->session->userdata('id')
            ?: 0
        );
    }

    private function _session_name()
    {
        return (string)(
            $this->session->userdata('nama')
            ?: $this->session->userdata('username')
            ?: ''
        );
    }

    private function _session_role()
    {
        return strtolower((string)$this->session->userdata('role') ?: '');
    }

//     public function seed_fix_user_hash()
// {
//     $this->load->database();

//     $users = $this->db
//         ->where("(hash IS NULL OR hash = '')", null, false)
//         ->get('tb_user')
//         ->result();

//     foreach ($users as $u) {
//         // hash pakai kolom password (123, 1234, dll)
//         $hash = password_hash($u->password, PASSWORD_BCRYPT);
//         $this->db->where('id', $u->id)->update('tb_user', ['hash' => $hash]);
//         echo "OK hash user: {$u->username}<br>";
//     }

//     echo "DONE.";
// }

// public function test_email()
// {
//     $this->load->library('email');

//     $this->email->from('fakethis889@gmail.com', 'Piket Daily Test');
//     $this->email->to('fakethis889@gmail.com');
//     $this->email->subject('Test Email dari CI');
//     $this->email->message('Hallo, ini email test dari aplikasi Piket Daily.');

//     if ($this->email->send()) {
//         echo "Email terkirim.";
//     } else {
//         echo "Gagal kirim:<br><pre>";
//         echo $this->email->print_debugger();
//         echo "</pre>";
//     }
// }


}
