<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Pembayaran_raftel extends CI_Controller
{
    const JENIS                 = 'raftel';
    const STATUS_SIAP_BERANGKAT = 3; // sesuaikan dgn StatusSiswa
    const STATUS_SISWA_DITOLAK  = 5;

    public function __construct()
    {
        parent::__construct();
        $this->load->model('Pembayaran_model',      'pembayaran');
        $this->load->model('PembayaranSiswa_model', 'ledger');
        $this->load->model('Biaya_model',           'biaya');
        $this->load->model('Siswa_model',           'siswa');
        $this->load->library(['form_validation','upload']);
        $this->load->helper(['form','url']);
    }

    /* ======================== AREA SISWA ======================== */

    /** Halaman siswa (form + ringkasan + riwayat) */
    public function index()
    {
        check_role([3]);

        // Pastikan siswa punya SiswaID; kalau belum → buatkan & simpan ke session.
        $siswa_id = $this->_ensure_siswa_id_for_current_user();
        if (!$siswa_id) show_error('Siswa tidak ditemukan.');

        $s = $this->siswa->get_by_id($siswa_id); // by SiswaID
        $negara     = $s->NegaraTujuan     ?? null;
        $perusahaan = $s->PerusahaanTujuan ?? null;
        $biaya      = $this->biaya->get_current(self::JENIS, $negara, $perusahaan);

        $total_bayar  = (float) $this->ledger->sum_nominal_by_siswa($siswa_id, self::JENIS);
        $total_biaya  = $biaya ? (float)$biaya->Nominal : 0.0;
        $sisa_tagihan = max(0, $total_biaya - $total_bayar);

        $data = [
            'title'        => 'Pembayaran Raftel',
            'riwayat'      => $this->pembayaran->get_by_siswa($siswa_id, self::JENIS),
            'ledger'       => $this->ledger->get_ledger_by_siswa($siswa_id, self::JENIS),
            'total_bayar'  => $total_bayar,
            'total_biaya'  => $total_biaya,
            'sisa_tagihan' => $sisa_tagihan,
        ];
        $this->load->view('pembayaran/raftel_index', $data);
    }

    /** Siswa submit pembayaran (status = Pending) */
    public function submit()
    {
        check_role([3]);

        $siswa_id = $this->_ensure_siswa_id_for_current_user();
        if (!$siswa_id) show_error('Siswa tidak ditemukan.');

        $this->form_validation->set_rules('JumlahBayar', 'Jumlah Bayar', 'required|numeric|greater_than[0]');
        $this->form_validation->set_rules('MetodeBayar', 'Metode Bayar', 'required');

        if ($this->form_validation->run() === FALSE) {
            $this->session->set_flashdata('error', validation_errors());
            return redirect('pembayaran_raftel');
        }

        // upload bukti (opsional)
        $config = [
            'upload_path'   => './uploads/bukti_transfer/',
            'allowed_types' => 'jpg|jpeg|png|pdf',
            'max_size'      => 4096,
            'encrypt_name'  => TRUE,
        ];
        if (!is_dir($config['upload_path'])) @mkdir($config['upload_path'], 0777, true);
        $this->upload->initialize($config);

        $bukti = null;
        if (!empty($_FILES['BuktiTransfer']['name'])) {
            if (!$this->upload->do_upload('BuktiTransfer')) {
                $this->session->set_flashdata('error', $this->upload->display_errors());
                return redirect('pembayaran_raftel');
            }
            $bukti = $this->upload->data('file_name');
        }

        $insert = [
            'SiswaID'       => $siswa_id,
            'JumlahBayar'   => (float)$this->input->post('JumlahBayar', true),
            'MetodeBayar'   => $this->input->post('MetodeBayar', true),
            'BuktiTransfer' => $bukti,
            'StatusBayar'   => 'Pending',
            // Model Pembayaran_model->get_by_siswa sudah handle nama kolom Jenis/JenisPembayaran
            'Jenis'         => self::JENIS,
        ];

        $this->db->trans_begin();
        $this->pembayaran->create_pending($insert);

        if ($this->db->trans_status() === FALSE) {
            $this->db->trans_rollback();
            $this->session->set_flashdata('error', 'Gagal menyimpan pembayaran.');
        } else {
            $this->db->trans_commit();
            $this->session->set_flashdata('success', 'Pembayaran dikirim. Menunggu verifikasi admin.');
        }
        return redirect('pembayaran_raftel');
    }

    /* ======================== AREA ADMIN ======================== */

    /** Tabel ringkasan seluruh siswa non-HSI (calon & aktif) untuk Raftel */
    public function admin_index()
    {
        check_role([1,2]);

        $list = $this->siswa->get_all_non_hsi(); // s.*, u.Email, u.Phone, ss.NamaStatus, l.NamaLPK
        $rows = [];

        foreach ($list as $s) {
            $biaya        = $this->biaya->get_current(self::JENIS, $s->NegaraTujuan ?? null, $s->PerusahaanTujuan ?? null);
            $nominal      = $biaya ? (float)$biaya->Nominal : 0.0;
            $sid          = $s->SiswaID;
            $total_bayar  = $sid ? (float)$this->ledger->sum_nominal_by_siswa($sid, self::JENIS) : 0.0;

            $rows[] = [
                'UserID'      => $s->UserID,
                'SiswaID'     => $sid,
                'NamaLengkap' => $s->NamaLengkap,
                'LPK'         => $s->NamaLPK,
                'Status'      => $s->NamaStatus ?: 'Calon',
                'TotalBiaya'  => $nominal,
                'TotalBayar'  => $total_bayar,
                'SisaTagihan' => max(0, $nominal - $total_bayar),
            ];
        }

        $data = [
            'title' => 'Admin - Pembayaran Raftel',
            'siswa' => $rows,
        ];
        $this->load->view('pembayaran/admin_raftel', $data);
    }

    /** Detail pembayaran satu siswa (id = UserID atau SiswaID) */
    public function detail($id = null)
    {
        check_role([1,2]);
        if (empty($id)) show_error('Parameter ID siswa tidak boleh kosong.');

        // coba sebagai SiswaID lalu fallback ke UserID (lebih robust)
        $s = $this->siswa->get_by_id($id, $id);
        if (!$s) show_error('Siswa tidak ditemukan.');

        $sid       = !empty($s->SiswaID) ? $s->SiswaID : null;
        $transaksi = $sid ? $this->pembayaran->get_by_siswa($sid, self::JENIS) : [];

        $data = [
            'title'     => 'Detail Pembayaran Raftel',
            'siswa'     => $s,
            'transaksi' => $transaksi,
        ];
        $this->load->view('pembayaran/detail_raftel', $data);
    }

    /** Approve satu transaksi (by PembayaranID) */
    public function approve($pembayaran_id)
    {
        check_role([1,2]);

        $row = $this->pembayaran->get_by_id($pembayaran_id);
        if (!$row || $row->StatusBayar !== 'Pending') {
            show_error('Pembayaran tidak ditemukan atau sudah diproses.');
        }
        if (isset($row->Jenis) && $row->Jenis !== self::JENIS) {
            show_error('Jenis pembayaran tidak sesuai.');
        }

        $this->db->trans_begin();
        $this->pembayaran->set_status($pembayaran_id, 'Approved');
        $this->ledger->insert_from_pembayaran($row, $this->session->userdata('UserID'), self::JENIS);
        $this->_recalculate_and_update_status($row->SiswaID);

        if ($this->db->trans_status() === FALSE) {
            $this->db->trans_rollback();
            $this->session->set_flashdata('error','Gagal approve pembayaran.');
        } else {
            $this->db->trans_commit();
            $this->session->set_flashdata('success','Pembayaran di-approve.');
        }
        return redirect('pembayaran_raftel/admin');
    }

    /** Reject satu transaksi (by PembayaranID) */
    public function reject($pembayaran_id)
    {
        check_role([1,2]);

        $alasan = $this->input->post('alasan'); // optional
        $row    = $this->pembayaran->get_by_id($pembayaran_id);

        if (!$row || $row->StatusBayar !== 'Pending') {
            show_error('Pembayaran tidak ditemukan atau sudah diproses.');
        }
        if (isset($row->Jenis) && $row->Jenis !== self::JENIS) {
            show_error('Jenis pembayaran tidak sesuai.');
        }

        $ok = $this->pembayaran->set_status($pembayaran_id, 'Rejected', $alasan);
        $this->session->set_flashdata($ok ? 'success':'error', $ok ? 'Pembayaran ditolak.' : 'Gagal menolak pembayaran.');
        return redirect('pembayaran_raftel/admin');
    }

    /** Reject siswa langsung dari list (bukan transaksi) */
    public function reject_siswa($user_id)
    {
        check_role([1]);
        $this->db->where('UserID', $user_id)->update('Siswa', ['StatusID' => self::STATUS_SISWA_DITOLAK]);
        $this->session->set_flashdata('success', 'Siswa ditolak dan dipindahkan ke menu Siswa Ditolak.');
        return redirect('pembayaran_raftel/admin');
    }

    /* ======================== UTIL ======================== */

    /** Recalc total bayar raftel vs biaya → update status jika lunas */
    private function _recalculate_and_update_status($siswa_id)
    {
        if (empty($siswa_id)) return;

        $s       = $this->siswa->get_by_id($siswa_id);
        $negara  = $s->NegaraTujuan     ?? null;
        $company = $s->PerusahaanTujuan ?? null;
        $biaya   = $this->biaya->get_current(self::JENIS, $negara, $company);

        $total = (float)$this->ledger->sum_nominal_by_siswa($siswa_id, self::JENIS);
        if ($biaya && $total >= (float)$biaya->Nominal) {
            $this->siswa->update_status($siswa_id, self::STATUS_SIAP_BERANGKAT);
        }
    }

    /**
     * Pastikan user saat ini punya SiswaID:
     * - Kalau sudah ada di session → pakai itu
     * - Kalau belum, cari row siswa by UserID
     * - Kalau row ada tapi SiswaID null → buat baru, update DB & set session
     * - Return SiswaID atau null kalau benar-benar tidak ada row siswa
     */
    private function _ensure_siswa_id_for_current_user()
    {
        // 1) dari session kalau sudah ada
        $sid = $this->session->userdata('SiswaID');
        if (!empty($sid)) return $sid;

        // 2) cari row siswa dari UserID
        $uid = $this->session->userdata('UserID');
        if (empty($uid)) return null;

        $s = $this->siswa->get_by_id(null, $uid); // by UserID
        if (!$s) return null;

        // 3) kalau belum punya SiswaID → auto-assign
        if (empty($s->SiswaID)) {
            $new = $this->_generate_siswa_id();
            $this->db->where('UserID', $uid)->update('Siswa', ['SiswaID' => $new]);
            $this->session->set_userdata('SiswaID', $new);
            return $new;
        }

        // 4) simpan ke session kalau ada
        $this->session->set_userdata('SiswaID', $s->SiswaID);
        return $s->SiswaID;
    }

    /** Generator SiswaID sederhana (prefix HSI + 5 digit) */
private function _generate_siswa_id()
{
    // Gunakan query khusus SQL Server (OFFSET FETCH)
    $row = $this->db->query("
        SELECT SiswaID 
        FROM Siswa
        WHERE SiswaID LIKE 'HSI%'
        ORDER BY SiswaID DESC
        OFFSET 0 ROWS FETCH NEXT 1 ROWS ONLY
    ")->row();

    $num = $row ? (int)substr($row->SiswaID, 3) + 1 : 1;
    return 'HSI' . str_pad($num, 5, '0', STR_PAD_LEFT);
}

}
